
; Turntable motor driver

; 32 samples to make up a sine wave
; 5-bit D/A converter;
; for 50Hz it's 625us per sample so x 32 = 20ms
; for 60Hz it should be 520.833333 but we use 520us per sample except for every 6th sample when a five is added
; So 520 x 5 + 525 = 3125us every six samples and so 96 samples (three 60Hz cycles) is 50ms. 50/3 = 16.666666ms = 60Hz
; runs at 20MHz for a 200ns resolution adjustment

; Fine tuning is plus or minus 200ns steps with a 12% maximum change either way (faster or slower) from 50Hz and 60Hz
   
	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _HS_OSC

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF


EEPROM0		equ	H'00'	; non-volatile storage MS 50Hz
EEPROM1		equ	H'01'	; non-volatile storage LS
EEPROM2		equ	H'02'	; non-volatile storage MS 60Hz
EEPROM3		equ	H'03'	; non-volatile storage LS

; bank 0 RAM 
STORE1			equ	H'20'	; delay counter	
STORE2			equ	H'21'	; delay counter
STORE3			equ	H'22'	;
DA_COUNT		equ	H'23'	; counter for digital to analog conversion

FIFTYADJ_MS	equ	H'24'	; 50Hz adjustment value
FIFTYADJ_LS	equ	H'25'	; 50Hz adjustment value
SIXTYADJ_MS	equ	H'26'	; 60Hz adjustment value	
SIXTYADJ_LS	equ	H'27'	; 60Hz adjustment value
PULSE_EXTN	equ	H'28' 	; for 60Hz extension to compensate for 8333ns 
TEMP_MS		equ	H'29'	; temporary register
TEMP_LS		equ	H'2A'	; temporary register

EXTRA			equ	H'2B'	; extend delay	

; all banks ram
W_TMP			equ	H'70'	; temporary store for w in interrupt
STATUS_TMP	equ	H'71'	; temporary store of status in interrupt 
INT_FLG			equ	H'72'	; used in EEPROM write

	ORG     2100
	DE		H'F3'	; 50Hz
	DE		H'CC'
	DE		H'F5'	; 60Hz (but with correction added for the 0.8333us non integer adjustment)
	DE		H'D6'	; 

; start program at memory 0
	org		0				; start at address 0000h
	goto	MAIN			; normal service routines from Reset vector
	org     	4				; interrupt vector 0004h, starts interrupt routine here (interrupt not used)
	goto	INTERRUPT	

;_________________________________________________________________________________________


; lookup table for 5-bit D/A
; 0 degrees 15,
; 11.25 degrees  19  (offset 4)
; 22.5 degrees  22  (offset 7)
; 33.75 degrees 24  (offset 9)
; 45 degrees 26  (offset 11)
; 56.25 degrees 28  (offset 13)
; 67.5 degrees 30  (offset 15)
; 78.75 degrees 31  (offset 16)
; 90 degrees 31  (offset 16)
; 
; 32 step (5-bit) sine generator (decimal) 16,19,21,24,26,28,30,31,31,31,30,28,26,24,21,19,16, 12,10,7,5,3,1,0,0,0,1,3,5,7,10,12,

SINE; look up table for 5-bit D/A
	addwf	PCL,f
; Note that circuit uses RA0 ms byte, RA4 ls byte so bit order 0-4 for what is returned in W is reversed for actual portA bits
	retlw	B'00000001'			;  (D16) B'00010000'
	retlw	B'00011001'			;  (D19) B'00010011'
	retlw	B'00010101'			;  (D21) B'00010101'
	retlw	B'00000011'			;  (D24) B'00011000'
	retlw	B'00001011'			;  (D26) B'00011010'
	retlw	B'00000111'			;  (D28) B'00011100'
	retlw	B'00001111'			;  (D30) B'00011110'

	retlw	B'00011111'			;  (D31) B'00011111'
	retlw	B'00011111'			;  (D31) B'00011111'		; peak maximum 90 degrees
	retlw	B'00011111'			;  (D31) B'00011111'

	retlw	B'00001111'			;  (D30) B'00011110'
	retlw	B'00000111'			;  (D28) B'00011100'
	retlw	B'00001011'			;  (D26) B'00011010'
	retlw	B'00000011'			;  (D24) B'00011000'
	retlw	B'00010101'			;  (D21) B'00010101'
	retlw	B'00011001'			;  (D19) B'00010011'
	retlw	B'00000001'			;  (D16) B'00010000'


	retlw	B'00000110'			;  (D12) B'00001100'
	retlw	B'00001010'			;  (D10) B'00001010'
	retlw	B'00011100'			;  (D07) B'00000111'
	retlw	B'00010100'			;  (D05) B'00000101'
	retlw	B'00011000'			;  (D03) B'00000011'
	retlw	B'00010000'			;  (D01) B'00000001'

	retlw	B'00000000'			;  (D00) B'00000000'
	retlw	B'00000000'			;  (D00) B'00000000'		; peak minimum 270 degrees
	retlw	B'00000000'			;  (D00) B'00000000'

	retlw	B'00010000'			;  (D01) B'00000001'
	retlw	B'00011000'			;  (D03) B'00000011'
	retlw	B'00010100'			;  (D05) B'00000101'
	retlw	B'00011100'			;  (D07) B'00000111'
	retlw	B'00001010'			;  (D10) B'00001010'
	retlw	B'00000110'			;  (D12) B'00001100'

; end lookup table

; ..............................................................................................................................
INTERRUPT
	
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0
	bcf		STATUS,RP1
	bsf		INT_FLG	,0		; set interrupt flag used as a signal that interrupt occurred

; adjust interrupt rate with counter

; if RB3 low, change mode to 60Hz

	btfss	PORTB,3
	goto	SIXTY_Hz

; set timer for 50Hz or adjusted value
	movf	FIFTYADJ_MS,w	; 50Hz adjustment value
	addwf	TMR1H,f
	movf	FIFTYADJ_LS,w	; 50Hz adjustment value			
	addwf	TMR1L,f
	btfsc	STATUS,C		; if carry increase MS byte
	incf		TMR1H,f
	bcf		PIR1,TMR1IF		; timer 1 flag

Hz_VALUE
	movf	DA_COUNT,w
	call		SINE
	movwf	PORTA
	incf		DA_COUNT,f
	movf	DA_COUNT,w
	xorlw	D'32'
	btfsc	STATUS,Z
	clrf		DA_COUNT		; end of sine cycle
	goto	EOI				; end of interrupt

SIXTY_Hz
; 60Hz

; set timer for 60Hz or adjusted value
	movf	SIXTYADJ_MS,w	; 60Hz adjustment value
	addwf	TMR1H,f
	movf	SIXTYADJ_LS,w	; 60Hz adjustment value			
	addwf	TMR1L,f
	btfsc	STATUS,C		; if carry increase MS byte
	incf		TMR1H,f
	bcf		PIR1,TMR1IF		; timer 1 flag

; add extra cycles every 6 samples to compensate for 520.833333us per sample to make up to integer
	decfsz	PULSE_EXTN,f
	goto	SKIP2
	movlw	D'6'
	movwf	PULSE_EXTN
	movlw	D'6'
	subwf	TMR1L,f
	btfss	STATUS,C		; if carry increase MS byte
	decf	TMR1H,f
	goto	CYCLE_X
SKIP2
; equalise position
	nop
	nop
	nop
	nop
	nop
	nop
	nop
CYCLE_X
	
	goto	Hz_VALUE

EOI
; end of interrupt reclaim w and status 

	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt
;________________________________________________________

MAIN

; set inputs/outputs
	movlw	B'00011110'	; set value 15 in 5-bit D/A (bit 0 most significant)
	movwf	PORTA
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'00001111'	; port B inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'11100000'	; outputs (0) and inputs (1)
	movwf	TRISA		; port A data direction register
	movlw	B'00000000'	; settings (pullups enabled)  
	movwf	OPTION_REG
	movlw	B'00000000' ; digital I/O
	movwf	ANSEL
	bcf		STATUS,RP0	; select memory bank 0
	clrf		T1CON
	bsf		T1CON,0		; allow timer 1 to run


	movlw	B'00011110'	; set value 15 in 5-bit D/A (bit 0 most significant)
	movwf	PORTA

; 1s startup allowing DC offset to settle
	movlw	D'5'
	movwf	EXTRA
START
	call		DELAY
	decfsz	EXTRA,f
	goto	START

	clrf		DA_COUNT
	
	movlw	D'6'
	movwf	PULSE_EXTN

	movlw	EEPROM0
	call		EEREAD
	movwf	FIFTYADJ_MS	; 50Hz adjustment value
	movlw	EEPROM1
	call		EEREAD
	movwf	FIFTYADJ_LS	; 50Hz adjustment value

	movlw	EEPROM2
	call		EEREAD
	movwf	SIXTYADJ_MS	; 60Hz adjustment value
	movlw	EEPROM3
	call		EEREAD
	movwf	SIXTYADJ_LS	; 60Hz adjustment value

; interrupt enable 
INTERRUPT_ENABLE
	bsf		STATUS,RP0	; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 enable
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PIR1,TMR1IF		; timer 1 flag
	bsf		INTCON,PEIE	; peripheral interrupt enable
	bsf		INTCON,GIE		; set global interrupt enable for above


WAIT_SW ; wait for a switch closure
; Switches

; if RB1 low slow rate
	btfss	PORTB,1
	goto	SLOWER
; if RB2 low increase rate
	btfss	PORTB,2
	goto	FASTER

; if RB0 low, change to default rate
	btfsc	PORTB,0
	goto	WAIT_SW

DEFAULT
; return to 50Hz or 60Hz exact values
; only selected rate changed 50 or 60Hz 

; Only write directly after interrupt
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG2
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG2
; check 50/60Hz
	btfss	PORTB,3		; test if 50 0r 60Hz
	goto	DEFAULT_60

; 50Hz
; store defaults 
	movlw	H'F3'
	movwf	FIFTYADJ_MS	; 50Hz adjustment value
	movlw	H'CC'
	movwf	FIFTYADJ_LS	; 50Hz adjustment value

	movlw	EEPROM0
	call		EEREAD		; sets EEADR
	movlw	H'F3'
	call		EEWRITE
; Only write straight after interrupt
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG3
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG3
	movlw	EEPROM1
	call		EEREAD
	movlw	H'CC'
	call		EEWRITE
	call		DELAY
	goto	WAIT_SW

DEFAULT_60
; 60Hz
	movlw	H'F5'
	movwf	SIXTYADJ_MS	; 60Hz adjustment value
	movlw	H'D6'
	movwf	SIXTYADJ_LS	; 60Hz adjustment value

	movlw	EEPROM2
	call		EEREAD
	movlw	H'F5'
	call		EEWRITE
; Only change straight after interrupt
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG4
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG4
	movlw	EEPROM3
	call		EEREAD
	movlw	H'D6'
	call		EEWRITE
	call		DELAY
	goto	WAIT_SW

SLOWER ; maximum change of 12%

; check 50/60Hz
	btfss	PORTB,3
	goto	SLOW_60	
; 50Hz mode

; adjust slower from 50Hz (or higher depending of Faster setting previously) to 44.6Hz approx 0.02Hz steps ~ .04%
; First step from 50Hz HF3CB	; 49.9840 Hz

; transfer to temp registers

	movf	FIFTYADJ_MS,w	; 50Hz adjustment value
	movwf	TEMP_MS
	movf	FIFTYADJ_LS,w	; 50Hz adjustment value
	movwf	TEMP_LS

; compare with decrease to H'F254' for 44.6Hz minimum
	movlw	H'F2'
	xorwf	TEMP_MS,w
	btfss	STATUS,Z
	goto	DEC_50
	movlw	H'54'
	xorwf	TEMP_LS,w
	btfsc	STATUS,Z
	goto	D1			; bypass decrease when at minimum
; do the decrease
DEC_50
	movf	TEMP_LS,f
	btfsc	STATUS,Z	; if ls byte is zero decrease ms byte
	decf	TEMP_MS,f
	decf	TEMP_LS,f

	goto	STORE50	; store values

SLOW_60
; 60Hz mode
; first step from 60Hz	H'F5D5'	; 59.97694Hz

; transfer to temp registers

	movf	SIXTYADJ_MS,w	; 60Hz adjustment value
	movwf	TEMP_MS
	movf	SIXTYADJ_LS,w	; 60Hz adjustment value
	movwf	TEMP_LS

; compare with decrease to H'F49B' for 53.57Hz minimum
	movlw	H'F4'
	xorwf	TEMP_MS,w
	btfss	STATUS,Z
	goto	DEC_60
	movlw	H'9B'
	xorwf	TEMP_LS,w
	btfsc	STATUS,Z
	goto	D2			; bypass decrease when at minimum

; do the decrease
DEC_60
	movf	TEMP_LS,f
	btfsc	STATUS,Z	; if ls byte is zero decrease ms byte
	decf	TEMP_MS,f
	decf	TEMP_LS,f

	goto	STORE60

FASTER; maximum change of 12%

; check 50/60Hz
	btfss	PORTB,3
	goto	FAST_60	
; 50Hz mode

; adjust faster from 50Hz (or from less depending on any previous slower setting)  to 56Hz 
; approx 0.02Hz steps ~ .04%
; First step from 50Hz H'F3CD	;  50.016Hz

; transfer to temp registers

	movf	FIFTYADJ_MS,w	; 50Hz adjustment value
	movwf	TEMP_MS
	movf	FIFTYADJ_LS,w	; 50Hz adjustment value
	movwf	TEMP_LS

; compare with increase to H'F51A' for 56Hz maximum
	movlw	H'F5'
	xorwf	TEMP_MS,w
	btfss	STATUS,Z
	goto	INC_50
	movlw	H'1A'
	xorwf	TEMP_LS,w
	btfsc	STATUS,Z
	goto	D1			; bypass increase when at maximum
; Do the increase
INC_50
	incf		TEMP_LS,f
	btfsc	STATUS,Z
	incf		TEMP_MS,f

STORE50
; Only change straight after interrupt
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG5
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG5
; Store in registers
	movf	TEMP_MS,w
	movwf	FIFTYADJ_MS	; 50Hz adjustment value
	movf	TEMP_LS,w
	movwf	FIFTYADJ_LS	; 50Hz adjustment value
; store in EEPROM

	movlw	EEPROM0
	call		EEREAD		; sets EEADR
	movf	TEMP_MS,w
	call		EEWRITE
; Only change straight after interrupt
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG6
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG6
	movlw	EEPROM1
	call		EEREAD
	movf	TEMP_LS,w
	call		EEWRITE
D1	call		DELAY
	goto	WAIT_SW
	
FAST_60
; 60Hz mode
; first step from 60Hz H'F5D7'	; 60.02304Hz

; transfer to temp registers

	movf	SIXTYADJ_MS,w	; 60Hz adjustment value
	movwf	TEMP_MS
	movf	SIXTYADJ_LS,w	; 60Hz adjustment value
	movwf	TEMP_LS

; compare with increase to H'F6EA' for 67.2Hz maximum
	movlw	H'F6'
	xorwf	TEMP_MS,w
	btfss	STATUS,Z
	goto	INC_60
	movlw	H'EA'
	xorwf	TEMP_LS,w
	btfsc	STATUS,Z
	goto	D2			; bypass increase when at maximum
; Do the increase
INC_60
	incf		TEMP_LS,f
	btfsc	STATUS,Z
	incf		TEMP_MS,f

STORE60
; Only change straight after interrupt
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG7
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG7
; Store in registers
	movf	TEMP_MS,w
	movwf	SIXTYADJ_MS	; 60Hz adjustment value
	movf	TEMP_LS,w
	movwf	SIXTYADJ_LS	; 60Hz adjustment value
; store in EEPROM

	movlw	EEPROM2
	call		EEREAD		; sets EEADR
	movf	TEMP_MS,w
	call		EEWRITE
; Only change straight after interrupt
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG8
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG8
	movlw	EEPROM3
	call		EEREAD
	movf	TEMP_LS,w
	call		EEWRITE
D2	call		DELAY
	goto	WAIT_SW

;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

; Subroutines


; subroutine to read EEPROM memory 

EEREAD
	bcf		STATUS,RP0	; select bank 0
	bsf 		STATUS,RP1	; select memory bank 2
	movwf 	EEADR			; indirect special function register
	bsf 		STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD		; read EEPROM
	bcf 		STATUS,RP0	; select memory bank 2
	movf	EEDATA,w		; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE
EEWRITE	
	clrf		INT_FLG			; set in interrupt
WAIT_FLAG1
	btfss	INT_FLG	,0		; when set, run write
	goto	WAIT_FLAG1
	bcf 		STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 		STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE		; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'			; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'			; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
	bsf 		INTCON,GIE		; enable interrupts
WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE			; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 		STATUS,RP0	; select memory bank 0
	return					; value written 

;  delay 
DELAY ; ~ 200ms for 20MHz crystal
	movlw	D'5'
	movwf	STORE3	
DELX
	movlw	D'255'		; delay extension
	movwf 	STORE2
DEL2_2
	movlw	D'255'		; delay routine
	movwf	STORE1	
DEL_CONT2_2
	decfsz	STORE1,f	; when 0, exit delay
	goto	DEL_CONT2_2
	decfsz	STORE2,f
	goto	DEL2_2	
	
	decfsz	STORE3,f
	goto	DELX		; delay extension
	return


	end
